#include <RTClib.h>

#include <Arduino.h>
#include <SigFox.h>
#include <ArduinoLowPower.h>
#include "DHT.h"
#include "Wire.h"
#include <SPI.h>
#include "SdFat.h"
#include "Sodaq_wdt.h"

//define PINs
int rainPin = 0;  //III 0 II 1 I 5
int Relais_Moist = 4; //III 4 II 4 I 4
int moisturePIN = A6; //III A6 II A6 I A6
int LightDetect = A3;  //III A3 II A3 I A4
int optocopler = 3; //switch for camera control, III 3 II 3 I 3
int Switch = 2;  //Switch for Pi, III 2 II 2 I 2

const uint8_t chipSelect = 7; //III 7 II 7 I 7

//#define WIRE Wire1
#define DHTPIN 14  //III 14 II 14 I 14
#define DHTTYPE DHT22   // DHT 22  (AM2302), AM2321
DHT dht(DHTPIN, DHTTYPE);

//camera parameters
int img_nbr = 1;
int thresholdLight = 40;  //III 100 II 40 I 40
int valAnalog_light;
int AutoSleepTimeCamera = 30; //in seconds
int DailyTimePictureHour = 9;

//pi parameters
int PiTurnOffMinutes = 6;

volatile bool israining = 0;
int rain_count_interval = 0;
float temp;
float hum;
int moist;

//rain intensity parameters (for Pi)
int timeRainTriggered;
int const maxConsiderationsIntens = 5;
int minutesRain[maxConsiderationsIntens] = {3,6,12,24,36};
int intensityRainThresh[maxConsiderationsIntens] = {5,8,15,20,25}; //tippings -> 1 tip = 0.2 mm
//int intensityRain[maxConsiderationsIntens] = {0,0,0,0,0};
unsigned int countRain = 0;
unsigned long start_rain;
unsigned long timeGoneSinceRainStarted;
unsigned long timeGoneSincePiTurnedOn = 0;
int hourStartRain;
unsigned long PiTurnedOnTime = 0;
bool triggerCameraDuetoMaxIntenseRainReached = false;
bool PiTurnedOnBool = false;
int sleepTime = 0;
int loopTriggerMaxCam  = 0;
bool tempInterrupt = false;
bool resetDuetoTime = false;

int messages_to_save = 4;
bool forced_send = 1;

RTC_DS3231 rtc;
DateTime jetzt;

typedef struct __attribute__ ((packed)) sigfox_message {
  uint16_t temp1;
  uint16_t temp2;
  uint8_t hum1;
  uint8_t hum2;
  uint8_t rain1;
  uint8_t rain2;
  uint16_t moist1;
  uint16_t moist2;
} SigfoxMessage;

SigfoxMessage msg;
SdFat sd;
File myFile;

void setup() {
    sodaq_wdt_enable(WDT_PERIOD_8X);
    
    Serial.begin(9600);

    //setup pins
    pinMode(LED_BUILTIN, OUTPUT);
    pinMode(rainPin, INPUT_PULLDOWN);
    pinMode(LightDetect, INPUT);
    pinMode(moisturePIN, INPUT);
    pinMode(optocopler, OUTPUT); digitalWrite(optocopler, LOW);
    pinMode(Relais_Moist, OUTPUT); digitalWrite(Relais_Moist, LOW);
    pinMode(Switch, OUTPUT); digitalWrite(Switch, LOW);
    
    rtc.begin();
      
    if (!sd.begin(chipSelect, SD_SCK_MHZ(50))) {
//      sd.initErrorHalt();
      blink(3,50);
    }

    sodaq_wdt_reset();
    if (rtc.lostPower()) {
      Serial.println("RTC lost power, lets set the time!");
      // following line sets the RTC to the date & time this sketch was compiled
      rtc.adjust(DateTime(F(__DATE__), F(__TIME__)));
      // This line sets the RTC with an explicit date & time, for example to set
      // January 21, 2014 at 3am you would call:
      // rtc.adjust(DateTime(2020, 3, 2, 14, 4, 0));
    }
  
    dht.begin();
    msg.hum1 = msg.hum2 = msg.rain1 = msg.rain2 = msg.temp1 = msg.temp2 = msg.moist1 = msg.moist2 = 0;
    blink(5,200);

    LowPower.attachInterruptWakeup(RTC_ALARM_WAKEUP, alarm, CHANGE);
    LowPower.attachInterruptWakeup(rainPin, itsraining, FALLING);

    //setup time for rain intensity
    jetzt = rtc.now();
    start_rain = jetzt.minute() * 60 + jetzt.second(); //in seconds
    hourStartRain = jetzt.hour();
    timeRainTriggered = start_rain;

    sodaq_wdt_disable();
    LowPower.sleep(sleeptime());
}

void loop() {
    sodaq_wdt_enable(WDT_PERIOD_8X);
    jetzt = rtc.now();

    unsigned long jetzt_minute = jetzt.minute();
    unsigned long jetzt_second = jetzt.second();
    unsigned long jetzt_seconds = jetzt_minute * 60 + jetzt_second;
    unsigned long jetzt_hour = jetzt.hour();
    
    //consider shift in hour, when working with seconds to measure passed time
    if (jetzt_hour != hourStartRain){
      if (jetzt_hour > hourStartRain){
        jetzt_seconds = jetzt_seconds + (abs(jetzt_hour - hourStartRain) * 60 * 60);
        timeRainTriggered = timeRainTriggered + (abs(jetzt_hour - hourStartRain) * 60 * 60);
        PiTurnedOnTime = PiTurnedOnTime + (abs(jetzt_hour - hourStartRain) * 60 * 60);
        start_rain = start_rain + (abs(jetzt_hour - hourStartRain) * 60 * 60);
      }
      else{
        jetzt_seconds = jetzt_seconds + (abs(jetzt_hour + 24 - hourStartRain) * 60 * 60);
        timeRainTriggered = timeRainTriggered + (abs(jetzt_hour + 24 - hourStartRain) * 60 * 60);
        PiTurnedOnTime = PiTurnedOnTime + (abs(jetzt_hour + 24 - hourStartRain) * 60 * 60);
        start_rain = start_rain + (abs(jetzt_hour + 24 - hourStartRain) * 60 * 60);
      }
    }    

    if (israining == 1 && resetDuetoTime == true) {
      start_rain = jetzt_seconds;
      hourStartRain = jetzt.hour();
      countRain = 0;
      resetDuetoTime = false;   
    }

    //check if rained within last 7 minutes to consider count for rain intensity and to reset hourStartRain
    if (triggerCameraDuetoMaxIntenseRainReached == false){
      if (jetzt_seconds - timeRainTriggered > 7 * 60){  //if not rained within last 7 minutes start from beginning
        if (israining == false){
          resetDuetoTime = true;
        }        
        start_rain = jetzt_seconds;
        hourStartRain = jetzt.hour();
        countRain = 0;        
      }
    }

    // to make daily picture (loopTriggerMaxCam has to be larger than for how long extra images taken due to high intenity rainfall, i.e. 10)
    int DailyTimePictureMinute = jetzt.minute();
    if (DailyTimePictureHour == jetzt_hour && DailyTimePictureMinute < 5){
      triggerCameraDuetoMaxIntenseRainReached = true;
      loopTriggerMaxCam = 11;
    }

    if (tempInterrupt == true){
      triggerCameraDuetoMaxIntenseRainReached = false;
    }
     
    if (israining == 1 || triggerCameraDuetoMaxIntenseRainReached == true) {      
        //Serial.println("woken up due to rain "); delay(20); Serial.print(rain_count_interval);
        blink(1, 1000);
        
        //check if enough light and trigger SLR camera
        delay(10);
        valAnalog_light = analogRead(LightDetect);
        if (valAnalog_light > thresholdLight){
          if (jetzt_seconds - timeRainTriggered > AutoSleepTimeCamera){
            trigger_camera(img_nbr);  //img_nbr + 1
          }
          else{
            trigger_camera(img_nbr); 
          }
        }        
        sodaq_wdt_reset();  

        //turn on Pi
        delay(50);
        if (PiTurnedOnBool == false){
          blink(2,100);
          blink(1,300);
          blink(2,100);
          if (valAnalog_light > thresholdLight){
            //Serial.println("Switching on Pi");
            log_pi(jetzt, true);
            sodaq_wdt_reset();
            delay(20);
            digitalWrite(Switch, HIGH); //pull switch wittypi
            PiTurnedOnTime = jetzt_seconds;
            PiTurnedOnBool = true;
          }
        }
         
        sodaq_wdt_reset();
        if (israining == 1){
          countRain++;           
          
          timeRainTriggered = jetzt_seconds;
          
          // check if rain intensity high enough to capture extra images
          timeGoneSinceRainStarted = (jetzt_seconds + 1) - start_rain; //in seconds          
          for (int i = 0; i < maxConsiderationsIntens; i = i + 1){
            delay(5);
            sodaq_wdt_reset();
            if (timeGoneSinceRainStarted >= minutesRain[i] * 60){ //check for different rain durations if above threshold
              //intensityRain[i] = (intensityRain[i] + ((countRain) / (timeGoneSinceRainStarted/60) * minutesRain[i]))/2;
              //Serial.print("Rain intensity "); Serial.println(intensityRain[i]);
              delay(5);
              //if (intensityRain[i] >= intensityRainThresh[i]){ 
              if (countRain > intensityRainThresh[i]){//check for different rain intensities if above threshold (for runoff)
                //Serial.print("Intensity thresh reached.");
                triggerCameraDuetoMaxIntenseRainReached = true;
                loopTriggerMaxCam = 0;
                blink(2,1000);
                sodaq_wdt_reset();
                break;
              }
            }
          } 
          
          // count rain, save rain to SD
          rain_count_interval++;
          log_rain(jetzt);
          israining = 0;
          sodaq_wdt_reset();          
        }        
    }

    else {  // no rain --> read sensors, save data        
        sodaq_wdt_reset();       
        delay(2000);
        temp = dht.readTemperature();
        hum = dht.readHumidity();
        digitalWrite(Relais_Moist, HIGH); //pull switch wittypi
        delay(2000);
        moist = averageAnalog(moisturePIN);
//        moist = analogRead(moisturePIN);
        delay(50);
        digitalWrite(Relais_Moist, LOW);                   
        sodaq_wdt_reset();
        
        log_Sensordata(jetzt, temp, hum, moist);
        
        if (jetzt.hour() == 0 && jetzt.minute() == 0) { //save extra messages at 24:00 before sending again
            messages_to_save = 4;
            forced_send = 1;
        }        
        
        if (jetzt.minute() % 10 == 0) { //send data every 10 minutes
            sodaq_wdt_reset();
            blink(2, 500);
            // Minute mit Teiler 10 --> Senden der Daten
            msg.temp2 = ((temp + 40) / 100) * 65535;
            msg.hum2 = (hum / 100) * 255;
            msg.moist2 = moist;
            msg.rain2 = rain_count_interval;
              
            rain_count_interval = 0;

            if (messages_to_save > 0) {
              if (forced_send == 1) {
                send_data(msg);
                forced_send = 0;
              }
              else if ((msg.rain1 + msg.rain2) > 0) {
                send_data(msg);          
              }
              else {
                forced_send = 1;
                messages_to_save--;
              }
            }
            else {
              send_data(msg);
            }            
        }

        else {  //save to sd every 5 minutes
            sodaq_wdt_reset();
            blink(4, 500);
            // Minute mit Teiler 5 --> nur setzen der msg.Teile
            msg.temp1 = ((temp + 40) / 100) * 65535;
            msg.hum1 = (hum / 100) * 255;
            msg.moist1 = moist;
            msg.rain1 = rain_count_interval;
            
            rain_count_interval = 0;
        }   
    }

    //if Pi turned on leave on for maximal 6 minutes and then turn off
    sodaq_wdt_reset();
    delay(10);
    timeGoneSincePiTurnedOn = (jetzt_seconds + 1) - PiTurnedOnTime;
    if (PiTurnedOnBool == true){
      if (timeGoneSincePiTurnedOn > PiTurnOffMinutes * 60){
        if (israining == 1 || triggerCameraDuetoMaxIntenseRainReached == true){  //turn off voltage to Pi and on again if turned on for at least 6 minutes and still raining
          sodaq_wdt_reset();  
          delay(20);
          digitalWrite(Switch, LOW);
          log_pi(jetzt, false);          
          delay(2000);
          digitalWrite(Switch, HIGH);
          PiTurnedOnTime = jetzt_seconds;
          log_pi(jetzt, true);
          sodaq_wdt_reset();
        }
        else  //turn off voltage to Pi if turned on for at least 6 minutes without rain
        {
          //Serial.println("Turning off Pi");
          log_pi(jetzt, false);
          sodaq_wdt_reset();  
          delay(20);
          digitalWrite(Switch, LOW);
          PiTurnedOnBool = false;
          PiTurnedOnTime = 0;
        }
      } 
    }

    if (tempInterrupt == true){ //true: shortly interrupted max intensity loop to allow for measuring weather parameters
      triggerCameraDuetoMaxIntenseRainReached = true;
      tempInterrupt = false;
    }

    //if rain intensity high enough trigger camera additionally at least for 5 minutes
    if (triggerCameraDuetoMaxIntenseRainReached == true){
      if (loopTriggerMaxCam > 10){  //to take images every 30 seconds for five minutes
        sleepTime = 0;
        triggerCameraDuetoMaxIntenseRainReached = false;
        loopTriggerMaxCam = 0;
      }
      else{
        loopTriggerMaxCam++;
        sleepTime = 30 * 1000;
        if (sleeptime() < sleepTime){
          sleepTime = 0;
          tempInterrupt = true;
        }        
      }
    }     

    log_debug(jetzt, jetzt_seconds, timeRainTriggered, triggerCameraDuetoMaxIntenseRainReached,
              loopTriggerMaxCam, PiTurnedOnTime, timeGoneSincePiTurnedOn, sleepTime, timeGoneSinceRainStarted,
              start_rain);
    sodaq_wdt_reset();
    sodaq_wdt_disable();
    if (sleepTime > 0) {
      LowPower.deepSleep(sleepTime);
    }
    else {
      LowPower.deepSleep(sleeptime());
    }
}


void alarm() {
}

void itsraining() { 
    israining = 1;
}

void log_rain(DateTime timestamp) {
    // save rain to SD
//    Serial.println("Writing to rain.txt...");
    myFile = sd.open("rain.txt", FILE_WRITE);
    myFile.print(timestamp.day());
    myFile.print(".");
    myFile.print(timestamp.month());
    myFile.print(".");
    myFile.print(timestamp.year());
    myFile.print(" ");
    myFile.print(timestamp.hour());
    myFile.print(":");
    myFile.print(timestamp.minute());
    myFile.print(":");
    myFile.println(timestamp.second());
    myFile.close();    
}

void log_debug(DateTime timestamp, unsigned long timeJetzt, unsigned long timeSinceRain,
               bool TrigCamMaxIntensRain, int loopTrigMaxCam, unsigned long PiTurnOnTime,
               unsigned long TimeGoneSincePiTurnOn, int timeSleep, unsigned long timeGonceSinceRainstart,
               unsigned long startRain) {
    // save rain to SD
//    Serial.println("Writing to rain.txt...");
    myFile = sd.open("debug.txt", FILE_WRITE);
    myFile.print(timestamp.hour());
    myFile.print(":");
    myFile.print(timestamp.minute());
    myFile.print(":");
    myFile.print(timestamp.second());
    myFile.print("; ");       
    myFile.print(timeJetzt);
    myFile.print("; ");
    myFile.print(timeSinceRain);
    myFile.print("; ");    
    myFile.print(TrigCamMaxIntensRain);
    myFile.print("; ");
    myFile.print(loopTrigMaxCam);
    myFile.print("; ");
    myFile.print(PiTurnOnTime);
    myFile.print("; ");
    myFile.print(TimeGoneSincePiTurnOn);
    myFile.print("; ");
    myFile.print(timeSleep);    
    myFile.print("; ");
    myFile.print(timeGonceSinceRainstart);    
    myFile.print("; ");
    myFile.println(startRain);      
    myFile.close();    
}


void log_Sensordata(DateTime timestamp, float temp, float hum, int moist) {
    // save rain to SD
//    Serial.println("Writing to weather.txt...");
    myFile = sd.open("weather.txt", FILE_WRITE);
    myFile.print(timestamp.day());
    myFile.print(".");
    myFile.print(timestamp.month());
    myFile.print(".");
    myFile.print(timestamp.year());
    myFile.print(" ");
    myFile.print(timestamp.hour());
    myFile.print(":");
    myFile.print(timestamp.minute());
    myFile.print(":");
    myFile.print(timestamp.second());
    myFile.print(";");
    myFile.print(temp);
    myFile.print(";");
    myFile.print(hum);
    myFile.print(";"); 
    myFile.println(moist);
    myFile.close();     
}

void log_pi(DateTime timestamp, bool PiTurnedOn) {
    // save pi state to SD
    myFile = sd.open("piLog.txt", FILE_WRITE);
    myFile.print(timestamp.day());
    myFile.print(".");
    myFile.print(timestamp.month());
    myFile.print(".");
    myFile.print(timestamp.year());
    myFile.print(" ");
    myFile.print(timestamp.hour());
    myFile.print(":");
    myFile.print(timestamp.minute());
    myFile.print(":");
    myFile.print(timestamp.second());
    myFile.print(";"); 
    myFile.println(PiTurnedOn);
    myFile.close();    
}

int sleeptime() {
    jetzt = rtc.now();
    int x = 1000 * (60 * (4 - (jetzt.minute() % 5)) + (60 - jetzt.second()));
    //int x = 1000 * (60 - jetzt.second());
    return x;
}

void blink(int x, int y) {
  for (int i=0; i < x; i++){
    digitalWrite(LED_BUILTIN, HIGH);   // turn the LED on (HIGH is the voltage level)
    delay(y);                       // wait for a second
    digitalWrite(LED_BUILTIN, LOW);    // turn the LED off by making the voltage LOW
    delay(y);    // wait for a second
  }
}

void trigger_camera(int img_nbrCam){
 for (int j=0; j<img_nbrCam; j++)
  {
    digitalWrite(optocopler, HIGH);
    delay(200);
    digitalWrite(optocopler, LOW);
    delay(2000);
    sodaq_wdt_reset();
  }
}

int averageAnalog(int PIN){
  int moistSum = 0;
  for (int k=0; k<10; k++)
  {
    int moistSingle = analogRead(PIN);
    delay(10);
    moistSum = moistSum + moistSingle;    
    sodaq_wdt_reset();
  }
  int moistAverage;
  moistAverage = moistSum / 10;
  return moistAverage;
}

void send_data(SigfoxMessage message) {
  sodaq_wdt_reset();
  SigFox.begin();
  // Wait at least 30ms after first configuration (100ms before)
  delay(50);
  SigFox.debug();
          
  SigFox.status();
  delay(1);
  SigFox.beginPacket();
  SigFox.write((uint8_t*)&message, 12);
  SigFox.endPacket();
  SigFox.end();
  sodaq_wdt_reset();
}
